###############################################################
# TPM 2.0 Sample AUX Deletion Script for BIOS based provisioning
# Copyright Intel Corporation 2015
# Last update: Oct 23, 2015
# 	Aug 2015: modified to work with PS index based on PS2 template
# --- *** This only works when the PS Index was created by BIOS (i,e, PS2 style)
#	OCT 2015 Added additional checks
#   Aug 2017 Added support for additional PS policies
###############################################################
##########################################################################
# Usage:                                                                 #
#     Tpm2DeleteAUX-BB.nsh   <alg>   <fileNamePrefix>   [-o]             #
#  Where:                                                                #
#      <alg> is  a supported hash algorithm                              #
#         Such as SHA1, SHA256, SHA384, SHA512, SM3                      #
#         <alg> will be substituted for %1 in Tpm2ProvTool command lines #
#      <fileNamePrefix> is the file name prefix you used to create the   #
#         def files - typically a project name or platform name          #
#        (e.g., MyFirst; Example; Purley)                                #
#         <fileNamePrefix> will be substituted for %2 in command lines   #
#         For example %2Ps2%1.iDef --> ExamplePs2Sha256.iDef             #
#                     %2Ps2%1.iDef --> MyFirstPs2Sha384.iDef             #
#      "-o" is an optional parameter, when specified, deletes original   #
#         AUX index handle(0x01800003) instead of the new (0x01C10102)   #
##########################################################################

echo -OFF 
# Copyrights:
# You may copy, distribute, and make derivatives of this script under the 
# following provisions:
#  You must not remove or modify the Intel copyright notice nor these requirements
#  The script must only be used to provision a TPM for use with Intel Technologies
#    For any other use you must get written permission from Intel Corporation
 
##########################################################################
# This script will:
# Check if the AUX Index exists 
# If so:
#	Delete the PS2 index 
#   Create a new PS2 Index
#   Write the PS2 Index to enable AUX Deletion
#   Delete the AUX NV Index 
#   Modify the PS Index to disable AUX Deletion
#	Lock the PS2 index
#
##########################################################################



##########################################################################
# Here is a list of files that are needed
## //User Modified Files -----------[example for %1=SHA256, %2=MyFirst]
## %2SecretOR%1.pDef                  MyFirstSecretOrSha256.pDef
## %2Ps2%1.iDef                       MyFirstPs2Sha256.iDef
## Intel Provided Files -----------------------------------------------------
## Aux%1.iDef                         AuxSha256.iDef
## Aux-o%1.iDef                       Aux-oSha256.iDef
## AuxPolicyOR%1.pDef                 AuxPolicyORSha256.pDef
## EmptyAuthPwSession.sDef
## getsec64.efi    -- part of Intel TXT Toolkit  
## Locality3&4.pDef
## Policy%1.sDef                      PolicySha256.sDef
## %1PsSha256.iDef
## PsDelAux.iDef
## PsDelAux-o.iDef
## UndefinespaceSpecial.pDef
##########################################################################
# This script echoes progress markers to the screen and also to
#   a log file (*.log) which also captures provisioning details 
#
#  Note that there are commands indicated by "# For testing on NEX"  
#    These never get executed when the script is run in the EFI environment
#     - they are for ease of manually testing using the NSH Exerciser
#     - Those lines may be deleted  
##########################################################################

#Check that Alg parameter is present
if ZZ%1  == ZZ   then
	echo Algorithm parameter missing
	echo Syntax: %0 <alg> <PlatformName>
	echo   Examples: 
	echo     %0 SHA256 Example 
	echo     %0 SHA384 MyFirst
	echo     %0 SHA512 Grantley
	echo     %0 SHA256 Old -o
	goto EOF
endif
# Check if alg parameter supported
if NOT EXIST Policy%1.sDef then 
	echo Algorithm not supported
	goto EOF
endif
# Check if platform parameter supported
if NOT EXIST %2Ps2%1.iDef then 
	echo Platform/Project Name not found (%2Ps2%1.iDef not found)
	goto EOF
endif
# Check if Param3 is correct
if NOT EXIST Aux%3%1.iDef then  
	echo Aux%3%1.iDef not found
	goto EOF
endif

# Now copy this file to the error log
echo TPM2 AUX Deletion Script %1 %2 %3 > Tpm2DeleteAux.log
if EXIST %0     then
	type %0     >> Tpm2DeleteAux.log
endif
if EXIST %0.nsh then 
	type %0.nsh >> Tpm2DeleteAux.log
endif

goto START
##########################################################################
# #####For testing on NEX #####
# The following commands do not need to be sent to a TPM on a 
#   target platform because BIOS starts the TPM  
#   -- rather they are here for testing on a simulator or TTV
#   --   using the NSH Exerciser
#
#  On Simulator start-up or after inserting a TPM in the TTV
Tpm2ProvTool  PowerUp      # for testing on NEX
Tpm2ProvTool  StartUpClear # for testing on NEX
#
# Before removing a TPM from the TTV
Tpm2ProvTool ShutdownClear # for testing on NEX
Tpm2ProvTool PowerDown     # for testing on NEX
##########################################################################

:START
echo ****** Deleting Aux Index *****
# Uses Session 0 for AuthPolicy
# Uses Session 1 for Empty AuthValue
# Note that session 0 uses "Don't Continue" so no need to flush on successful use

echo **** Checking if AUX index exists
echo **** Checking if AUX index exists >> Tpm2DeleteAux.log
Tpm2ProvTool NvReadPublic Aux%3%1.iDef >> Tpm2DeleteAux.log
if not %lasterror% == 0 then
	echo Aux Index does not exist - nothing to delete
	goto EOF
endif

# Start a PW Session to use for PhAuth
echo **** Starting PW Session for PlatformAuth and IndexAuth
echo **** Starting PW Session for PlatformAuth and IndexAuth >> Tpm2DeleteAux.log
Tpm2ProvTool StartSession EmptyAuthPwSession.sDef 1 >> Tpm2DeleteAux.log

# This script requires that PlatformAuth be EMPTY in order to delete NV Indexes
# Lets check if PlatformAuth is EMPTY by attemping to change it from EMPTY to EMPTY
Tpm2ProvTool HierarchyChangeAuth PH EmptyAuth.dat 1 >> Tpm2DeleteAux.log
if not %lasterror% == 0 then
	echo PlatformAuth is not EMPTY
	echo Try running ResetPlatformAuth.nsh %1 %2
	goto ERROR
endif

echo Deleting the AUX Index requires re-writing PS Index
echo Rewriting a PS2 Index requires deleting PS Index

echo **** Checking if PS index exists
echo **** Checking if PS index exists >> Tpm2DeleteAux.log
Tpm2ProvTool NvReadPublic %2PS2%1.iDef >> Tpm2DeleteAux.log
if not %lasterror% == 0 then
	echo PS Index does not exist - nothing to delete
	goto PSCREATE
endif

# Delete PS2 index
echo *** Satisfying PS2 authPolicy (PS Index Deletion)
echo *** Starting Policy Session for PS Index auth
echo *** Starting Policy Session for PS Index auth >> Tpm2DeleteAux.log
Tpm2ProvTool StartSession Policy%1.sDef 0 >> Tpm2DeleteAux.log
	
# Satisfy the PS authPolicy
# There are 4 possible policy styles
#	A. single branch with 1 assertion:    PolCommandCode(NV_UndefinespaceSpecial)
#	B. multiple branches with 1 of them:  PolCommandCode(NV_UndefinespaceSpecial)
#	C. single branch with 2 assertions:   PolOR(0,SecretValue) && PolCommandCode(NV_UndefinespaceSpecial)
#	D. multiple branches with 1 of them:  PolOR(0,SecretValue) && PolCommandCode(NV_UndefinespaceSpecial)

# Let's try A & B first
	
echo *** Policy Command Code (TPM_CC_NV_UndefineSpace)
echo *** Policy Command Code (TPM_CC_NV_UndefineSpace) >> Tpm2DeleteAux.log
Tpm2ProvTool AssertPolicy UnDefineSpaceSpecial.pDef 0 >> Tpm2DeleteAux.log
# Tpm2ProvTool StorePolDigest 0 6 >> Tpm2DeleteAux.log

# this is all that is needed for Style A (PS2) - so let's try
echo *** Execute UndefineSpecial for PS Index
echo *** Execute UndefineSpecial for PS Index >> Tpm2DeleteAux.log
Tpm2ProvTool NvUndefineSpecial %2Ps%1.iDef 0 1 >> Tpm2DeleteAux.log
if %lasterror% == 0 then
    goto PSCREATE
endif

# you get here if that did not work,  So lt's assume style B
#  so send the final PolicyOR assertion needed for PS1 and try again

echo *** Policy OR (Branch 1, Branch 2, Branch 3)
echo *** Policy OR (Branch 1, Branch 2, Branch 3) >> Tpm2DeleteAux.log
Tpm2ProvTool AssertPolicy %2PsFinalOR%1.pDef 0 >> Tpm2DeleteAux.log
# Tpm2ProvTool StorePolDigest 0 6 >> Tpm2DeleteAux.log

echo *** Execute UndefineSpecial for PS Index (Style B)
echo *** Execute UndefineSpecial for PS Index (Style B) >> Tpm2DeleteAux.log
Tpm2ProvTool NvUndefineSpecial %2Ps%1.iDef 0 1 >> Tpm2DeleteAux.log
if %lasterror% == 0 then
    goto PSCREATE
endif

# OK, Let's try style C	
# you get here if that did not work,  So lt's assume style C
# First we have to reset the Policy Session
echo **** Restart session
Tpm2ProvTool PolicyRestart 0 >> Tpm2DeleteAux.log

echo *** Policy OR (0, %2PsSecret%1)
echo *** Policy OR (0, %2PsSecret%1) >> Tpm2DeleteAux.log
Tpm2ProvTool AssertPolicy %2SecretOR%1.pDef 0 >> Tpm2DeleteAux.log
#Tpm2ProvTool StorePolDigest 0 6 >> Tpm2DeleteAux.log

echo *** Policy Command Code (TPM_CC_NV_UndefineSpace)
echo *** Policy Command Code (TPM_CC_NV_UndefineSpace) >> Tpm2DeleteAux.log
Tpm2ProvTool AssertPolicy UnDefineSpaceSpecial.pDef 0 >> Tpm2DeleteAux.log
# Tpm2ProvTool StorePolDigest 0 6 >> Tpm2DeleteAux.log

# this is all that is needed for PS2 - so let's see
echo *** Execute UndefineSpecial for PS Index
Tpm2ProvTool NvUndefineSpecial %2Ps%1.iDef 0 1 >> Tpm2DeleteAux.log
if %lasterror% == 0 then
    goto PSCREATE
endif

# you get here if that did not work, 
#  so send the final PolicyOR assertion needed for PS1 and try again

echo **** Policy OR (Branch 1, Branch 2, Branch 3)
echo **** Policy OR (Branch 1, Branch 2, Branch 3) >> Tpm2DeleteAux.log
Tpm2ProvTool AssertPolicy %2PsFinalOR%1.pDef 0 >> Tpm2DeleteAux.log
Tpm2ProvTool StorePolDigest 0 6 >> Tpm2DeletePS.log

echo **** UndefineSpecial PS Index
echo **** UndefineSpecial PS Index >> Tpm2DeleteAux.log
Tpm2ProvTool NvUndefineSpecial %2Ps%1.iDef 0 1 >> Tpm2DeleteAux.log
if not %lasterror% == 0 then
    echo Requires PlatformAuth be EMPTY and Satisfying index authPolicy
	echo If PlatformAuth is not EMPTY, then first run ResetPlatformAuth.nsh %1 %2
	goto ERROR		# Didn't satisfy PS Index authPolicy
endif

echo  *******    PS Index Deleted Successfully    *******


:PSCREATE
# Create a new PS2 index
echo ***** Creating PS2 Index *****
echo **** NV_DefineSpace for PS Index
echo **** NV_DefineSpace for PS Index >> Tpm2DeleteAux.log
Tpm2ProvTool NvDefineSpace %2Ps2%1.iDef 1 >> Tpm2DeleteAux.log
if not %lasterror% == 0 then
    echo Error creatig the PS Index 
	goto ERROR		# Didn't create the NV Index
endif


:PSREWRITE1
echo Write PS Policy and set Aux-Delete-Enable

echo *** Writing PS NV Data
echo *** Writing PS NV Data >> Tpm2DeleteAux.log
Tpm2ProvTool NvWrite %2Ps2%1.iDef 1 >> Tpm2DeleteAux.log
if not %lasterror% == 0 then
	echo Error writing the PS Index
	goto ERROR		# Unknown error  -- session/bad attributes
endif
echo *** Writing PS Control Word
echo *** Writing PS Control Word >> Tpm2DeleteAux.log
Tpm2ProvTool NvWrite PsDelAux%3.iDef 1 >> Tpm2DeleteAux.log
if not %lasterror% == 0 then
	echo Error writing the PS Index
	goto ERROR		# Unknown error  -- session/bad attributes
endif

:AUXDELETE
#echo ***** Deleting AUX Index *****
echo "ServerGetSec -l ent -fn res"
echo "ServerGetSec -l ent -fn res" >> Tpm2DeleteAux.log
getsec64 -l ent -fn res >> Tpm2DeleteAux.log
if not %lasterror% == 0 then
	echo Unable to delete Aux index
	goto ERROR		# Didn't delete AUX Index 
endif
goto PSREWRITE2
##########################################################################
# Delete AUX on Simulator or TTV
# **** Start Policy Session 
Tpm2ProvTool StartSession Policy%1.sDef 0  # For testing on NEX
# **** Locality3&4 Assertion 
Tpm2ProvTool AssertPolicy Locality3&4.pDef 0  # For testing on NEX
# **** CommandCode assertion
Tpm2ProvTool AssertPolicy UndefineSpaceSpecial.pDef 0  # For testing on NEX
# **** PolicyOR assertion
Tpm2ProvTool AssertPolicy AuxPolicyORSha256.pDef 0  # For testing on NEX
# **** change to locality 3 
Tpm2ProvTool SetActiveLocality 3 # For testing on NEX
# **** UndefineSpaceSpecial for AUX
Tpm2ProvTool NvUndefineSpecial Aux%3%1.iDef 0 1  # For testing on NEX
if not %lasterror% == 0 then
	echo Cannot delete AUX index
    Tpm2ProvTool SetActiveLocality 0
	goto ERROR		
endif
Tpm2ProvTool SetActiveLocality 0 # For testing on NEX

##########################################################################
:PSREWRITE2

echo *** Writing PS NV Data
echo *** Writing PS NV Data >> Tpm2DeleteAux.log
Tpm2ProvTool NvWrite %2Ps2%1.iDef 1 >> Tpm2DeleteAux.log
if not %lasterror% == 0 then
	echo Unable to rewrite PS Index
	goto PSERROR		# Didn't write Index 
endif

#    Write Protect PS Index
echo Writing PS Index
Tpm2ProvTool NVWriteLock %2Ps2%1.iDef 1 >> Tpm2DeleteAux.log
if not %lasterror% == 0 then
	echo PS WriteLock failed
	goto PSERROR		
endif

# Need to check to make sure AUX Deletion was really successful
echo **** Checking if AUX index was deleted
echo **** Checking if AUX index was deleted >> Tpm2DeleteAux.log
Tpm2ProvTool NvReadPublic Aux%3%1.iDef >> Tpm2DeleteAux.log
if %lasterror% == 0 then
	echo Aux Index was not deleted
	goto ERROR		# Index still exists
endif

:DONE
echo  ***************************************************
echo  ***********   AUX Deleted Successfully  ***********
echo  ***************************************************

goto EOF

:ERROR
echo  ***************************************************
echo  *************** ERROR Deleting AUX  ***************
echo  ***************************************************
Tpm2ProvTool FlushSession 0 >> Tpm2DeleteAux.log

goto EOF

:PSERROR
echo  ***************************************************
echo  *********** ERROR ReWriting/Locking PS  ***********
echo  ***************************************************

goto EOF

:EOF

